import { TrendingUp, Eye, EyeOff, PieChart } from "lucide-react";
import { useState } from "react";

const portfolioAssets = [
  { symbol: "BTC", name: "Bitcoin", amount: 0.156, value: 10518.67, change: 2.34, color: "#F7931A" },
  { symbol: "ETH", name: "Ethereum", amount: 2.45, value: 8469.11, change: -1.23, color: "#627EEA" },
  { symbol: "SOL", name: "Solana", amount: 12.5, value: 2230.63, change: 5.67, color: "#00FFA3" },
  { symbol: "AVAX", name: "Avalanche", amount: 28.3, value: 1197.09, change: 3.21, color: "#E84142" },
];

const PortfolioSection = () => {
  const [hideBalance, setHideBalance] = useState(false);
  const totalValue = portfolioAssets.reduce((sum, asset) => sum + asset.value, 0);
  const totalChange = 2.15;

  return (
    <section className="py-20 relative">
      <div className="container mx-auto px-4">
        <div className="grid lg:grid-cols-3 gap-8">
          {/* Portfolio Summary */}
          <div className="lg:col-span-2">
            <div className="glass rounded-2xl p-6">
              <div className="flex items-center justify-between mb-6">
                <div>
                  <div className="flex items-center gap-2 text-muted-foreground mb-1">
                    <span>Balance total</span>
                    <button onClick={() => setHideBalance(!hideBalance)}>
                      {hideBalance ? (
                        <EyeOff className="w-4 h-4" />
                      ) : (
                        <Eye className="w-4 h-4" />
                      )}
                    </button>
                  </div>
                  <div className="text-4xl font-bold text-foreground">
                    {hideBalance ? "••••••" : `$${totalValue.toLocaleString('en-US', { minimumFractionDigits: 2 })}`}
                  </div>
                </div>
                <div className="flex items-center gap-2 px-4 py-2 rounded-lg bg-success/20 text-success">
                  <TrendingUp className="w-4 h-4" />
                  <span className="font-semibold">+{totalChange}%</span>
                </div>
              </div>

              {/* Assets List */}
              <div className="space-y-4">
                <div className="text-sm text-muted-foreground font-medium mb-2">
                  Tus activos
                </div>
                {portfolioAssets.map((asset, index) => (
                  <div
                    key={asset.symbol}
                    className="flex items-center justify-between p-4 rounded-xl bg-secondary/50 hover:bg-secondary transition-colors animate-slide-in"
                    style={{ animationDelay: `${index * 0.1}s` }}
                  >
                    <div className="flex items-center gap-3">
                      <div
                        className="w-10 h-10 rounded-full flex items-center justify-center text-white font-bold"
                        style={{ backgroundColor: asset.color }}
                      >
                        {asset.symbol.charAt(0)}
                      </div>
                      <div>
                        <div className="font-semibold text-foreground">{asset.name}</div>
                        <div className="text-sm text-muted-foreground">
                          {asset.amount} {asset.symbol}
                        </div>
                      </div>
                    </div>
                    <div className="text-right">
                      <div className="font-semibold text-foreground">
                        {hideBalance ? "••••" : `$${asset.value.toLocaleString('en-US', { minimumFractionDigits: 2 })}`}
                      </div>
                      <div className={`text-sm ${asset.change >= 0 ? 'text-success' : 'text-destructive'}`}>
                        {asset.change >= 0 ? '+' : ''}{asset.change}%
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            </div>
          </div>

          {/* Allocation Chart */}
          <div className="glass rounded-2xl p-6">
            <div className="flex items-center gap-2 mb-6">
              <PieChart className="w-5 h-5 text-primary" />
              <span className="font-semibold text-foreground">Distribución</span>
            </div>

            {/* Simple Visual Chart */}
            <div className="relative w-48 h-48 mx-auto mb-6">
              <svg viewBox="0 0 100 100" className="w-full h-full -rotate-90">
                {portfolioAssets.reduce((acc, asset, index) => {
                  const percentage = (asset.value / totalValue) * 100;
                  const previousTotal = acc.total;
                  acc.total += percentage;
                  acc.elements.push(
                    <circle
                      key={asset.symbol}
                      cx="50"
                      cy="50"
                      r="40"
                      fill="transparent"
                      stroke={asset.color}
                      strokeWidth="20"
                      strokeDasharray={`${percentage * 2.51} ${251 - percentage * 2.51}`}
                      strokeDashoffset={-previousTotal * 2.51}
                      className="transition-all duration-500"
                      style={{ animationDelay: `${index * 0.1}s` }}
                    />
                  );
                  return acc;
                }, { elements: [] as JSX.Element[], total: 0 }).elements}
              </svg>
              <div className="absolute inset-0 flex items-center justify-center">
                <div className="text-center">
                  <div className="text-2xl font-bold text-foreground">4</div>
                  <div className="text-xs text-muted-foreground">Activos</div>
                </div>
              </div>
            </div>

            {/* Legend */}
            <div className="space-y-3">
              {portfolioAssets.map((asset) => (
                <div key={asset.symbol} className="flex items-center justify-between">
                  <div className="flex items-center gap-2">
                    <div
                      className="w-3 h-3 rounded-full"
                      style={{ backgroundColor: asset.color }}
                    />
                    <span className="text-sm text-foreground">{asset.symbol}</span>
                  </div>
                  <span className="text-sm text-muted-foreground">
                    {((asset.value / totalValue) * 100).toFixed(1)}%
                  </span>
                </div>
              ))}
            </div>
          </div>
        </div>
      </div>
    </section>
  );
};

export default PortfolioSection;
